/*!---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
import { StackFrame, StructuredError, PackageInfo, pathMode } from "./";
import { ErrorReporter } from "../error-reporter-base";
export interface StackProcessorOptions {
    appRoot: string;
    mode?: pathMode;
    errorReporter?: ErrorReporter;
    packageInfo: PackageInfo;
}
export declare class StackProcessor {
    private appRoot;
    private path;
    private doFileIO;
    private _errorReporter;
    private _packageInfo;
    constructor(options: StackProcessorOptions);
    /**
     * Parses out the first line of a standard Error Stack trace, which consists of the type and message, if present.
     */
    parseFirstLine(firstLine: string): StructuredError;
    /**
     * Parses out an actual stack frame line from a stack trace.
     */
    parseStackLine(inputLine: string, getPackageInfo?: boolean): StackFrame;
    cleanEvalFrames(input: string): string;
    /**
     * Parses out the path to a file containing code that is on the stack.
     * It does not strip the path at all.
     */
    parseCodeLocation(input: string): StackFrame;
    /**
     * given a stack frame with a full path, finds the info of the package associated with the file.
     * Also strips the path down to the relative path.
     */
    getPackageInfo(filePath: string): {
        package: PackageInfo;
        strippedPath: string;
    };
}
/**
 * Given a string containing a path, will find a username, if present, and remove all instances of it.
 */
export declare function removeUserName(input: string): string;
/**
 * Outputs a very thorough stack trace in the roughly format the Watson would expect it in.
 * PII is removed.
 */
export declare function structuredToSemiWatson(input: StructuredError): string;
/**
 * converts a structured error object into a flat string matching the format of a standard stack trace,
 * but with stripped paths and other small changes compared to a standard v8 Stack.
 */
export declare function structuredToStandard(input: StructuredError): string;
/**
 * Converts a StackFrame object into a well formatted string.
 */
export declare function frameToString(frame: StackFrame): string;
